;
;        Copyright (c) 2005 AileronOS developers group
;        All rights reserved
;        See eula.txt for more information
;
;    File:
;
;        boot\hdd\boothdd.asm
;
;    Abstract:
;
;        Aileron operating system bootloader for HDD
;
;    Created:
;
;        Vadim Ushakov, 2005
;
;    Revision:
;
;        Vlad Maslakov:
;        - destination address modified to 0x600
;

use16
org 0x7C00

jmp	start
nop
bsOemName             db "        " ; 0x03

; Start of BPB.
BPB_BytsPerSec        dw ? ; 0x0B
BPB_SecPerClus        db ? ; 0x0D
BPB_RsvdSecCnt        dw ? ; 0x0E
BPB_NumFATs           db ? ; 0x10
BPB_RootEntCnt        dw ? ; 0x11
BPB_TotSec16          dw ? ; 0x13
BPB_Media             db ? ; 0x15
BPB_FATSz16           dw ? ; 0x16
BPB_SecPerTrk         dw ? ; 0x18
BPB_NumHeads          dw ? ; 0x1A
BPB_HiddSec           dd ? ; 0x1C
BPB_TotSec32          dd ? ; 0x20

; FAT32 cpecific
BPB_FATSz32           dd ? ; 0x24
BPB_ExtFlags          dw ? ; 0x28
BPB_FSVer             dw ? ; 0x2a
BPB_RootClus          dd ? ; 0x2c
BPB_FSInfo            dw ? ; 0x30
BPB_BkBootSec         dw ? ; 0x32
BPB_Reserved          rb 12; 0x34
; End of BPB.

BS_DrvNum             db ? ; 0x40
BS_Reserved1          db ? ; 0x41
BS_BootSig            db ? ; 0x42
BS_VolID              rb 4 ; 0x43
BS_VolLab             db "           " ; 0x47
BS_FilSysType         db "FAT32   "    ; 0x52

struc B{

 .none              rb 3

 .bsOemName         db "        " ; 0x03
 .BytsPerSec        dw ? ; 0x0B
 .SecPerClus        db ? ; 0x0D
 .RsvdSecCnt        dw ? ; 0x0E
 .NumFATs           db ? ; 0x10
 .RootEntCnt        dw ? ; 0x11
 .TotSec16          dw ? ; 0x13
 .Media             db ? ; 0x15
 .FATSz16           dw ? ; 0x16
 .SecPerTrk         dw ? ; 0x18
 .NumHeads          dw ? ; 0x1A
 .HiddSec           dd ? ; 0x1C
 .TotSec32          dd ? ; 0x20

 ; FAT32 specific
 .FATSz32           dd ? ; 0x24
 .ExtFlags          dw ? ; 0x28
 .FSVer             dw ? ; 0x2a
 .RootClus          dd ? ; 0x2c
 .FSInfo            dw ? ; 0x30
 .BkBootSec         dw ? ; 0x32
 .Reserved          rb 12; 0x34
 ; End of BPB

 .DrvNum             db ? ; 0x40
 .Reserved1          db ? ; 0x41
 .BootSig            db ? ; 0x42
 .VolID              rb 4 ; 0x43
 .VolLab             db "           " ; 0x47
 .FilSysType         db "FAT32   "    ; 0x52

 .FirstLogSec        dd ?
 .PrevFATSec         dd ?

 .A                  dw ?
}
virtual at 0
 B B
end virtual

start:
	; setup segment register
	xor	eax, eax
	mov	es, ax
	mov	ds, ax
	mov	ss, ax

	mov	bp, 0x7c00 ; set pointer to the work structure
	mov	sp, bp     ; setup stack

	; remember device number we loading from
	mov	[bp + B.DrvNum], dl

	; ensure that it is not FAT12 or FAT16
	cmp	[bp + B.FATSz16], ax
	jnz	Error_FS_Err

	; checking FAT32 version
	cmp	[bp + B.FSVer], ax
	jnz	Error_FS_Err


	; calculating field with clusters
	mov	al, [bp + B.NumFATs]
	mov	ecx, [bp + B.FATSz32]
	mul	ecx
	add	eax, [bp + B.HiddSec]
	movzx	edx, [bp + B.RsvdSecCnt]
	add	eax, edx
	mov	[bp + B.FirstLogSec], eax

	mov	eax, [bp + B.RootClus]

	; this cycle looks over all clusters in the root directory
	.l_loop01:
	push	eax

		; ,     
		cmp	eax, 0x0ffffff8
		jnb	Error_Loader_Not_Found

		sub	eax, 2 ; 2 first clusters has no physical representation
		jb	Error_FS_Err ; cluster number must be greater than 1
		movzx	ebx, [bp + B.SecPerClus]
		mov	si, bx
		mul	ebx
		add	eax, [bp + B.FirstLogSec]

		; the cycle looks over all sectors in the cluster
		.l_loop02:

			; read sector of the root directory
			mov	bx, 0x8200
			mov	di, bx
			mov	cx, 1
			call	ReadSector

			; the cycle looks over all records in the root directory
			.l_loop03:

				; error if loader not found
				cmp	[di], ch
				jz	Error_Loader_Not_Found

				; comparing file name with the bootloader name
				mov	cx, 11
				push	si
				mov	si, ProgramName
				rep	cmpsb
				pop	si
				jz  found

				add	di, cx
				add	di, 0x15
			cmp	di, 0x8200 + 0x200
			jb	.l_loop03

		inc	eax
		dec	si
		jnz	.l_loop02

	pop	eax
	call	GetNextClus
	jmp	.l_loop01


found:

	; get bootloader first cluster
	mov	ax, [di + 9]    ; high part of the cluster number
	mov	di, [di + 0x0f] ; low part of the cluster number
	shl	eax, 0x10
	mov	ax, di

	; checking file size
	cmp	eax, 0x0ffffff8
	jnb	Error_Loader_Not_Found

    ; setting destination to 0x600
	mov	[bp + B.A], word 0x60

	; the cycle reads file into memory
	.loop01:

		cmp	eax, 0x0ffffff8
		jnb	all_ok

		push	eax

		; calculating sector number corresponding cluster begin
		sub	eax, 2
		jb	Error_FS_Err
		movzx	ecx, [bp + B.SecPerClus]
		mul	ecx
		add	eax, [bp + B.FirstLogSec]

		; read sectors of one cluster
		push	es
		mov	es, [bp + B.A]
		mov	bx, 0
		call	ReadSector
		pop	es

		; correcting B.A
		shl	cx, 5
		add	[bp + B.A], cx

		pop	eax
	call	GetNextClus
	jmp	.loop01

Error:
	mov	ah, 0x7d
	mov	si, ax

	.l_printloop:
	lodsb
	test	al, al
	jz	.l_waitkey_and_reboot
	mov	ah, 0x0e
	mov	bx, 7
	int	10h
	jmp	.l_printloop

	.l_waitkey_and_reboot:
	cbw
	int	0x16
	int	0x19

Error_Loader_Not_Found:
	mov	al, msg_Loader_Not_Found and 0xff
	jmp	Error
Error_IO_Err:
	mov	al, msg_IO_Err and 0xff
	jmp	Error
Error_FS_Err:
	mov	al, msg_FS_Err and 0xff
	jmp	Error


;
;    Function:
;
;        ReadSector
;
;    Description:
;
;        Reads sector into memory
;
;    Input:
;
;        eax   - sector number
;        es:bx - output buffer
;        cx    - sectors count
;
ReadSector:
	pushad

	push	dword 0 ; high 32 bits of LBA
	push	eax     ; low 32 bits of LBA
	push	es      ; buffer segment
	push	bx      ; segment offset
	push	cx      ; sectors count
	push	word 0x0010 ; package count

	; calling service 0x42 (Extended Read)
	mov	ah, 0x42
	mov	si, sp
	mov	dl, [bp + B.DrvNum]
	int	0x13

	; reading error?
	jc	Error_IO_Err

	add	sp, 4 * 4
	popad
	retn


;
;    Function:
;
;        GetNextClus
;
;    Description:
;
;        The function get from current cluster number next cluster number
;
;    Input:
;
;        eax - current cluster number
;
;    Output:
;
;        eax - next cluster number
;
GetNextClus:
	; offset in FAT
	shl	eax, 2

	; read data from FAT
	call	ReadFATSec
	mov	eax, [0x8000 + bx]
	and	eax, 0x0fffffff

	retn

;
;    Function:
;
;        ReadFATSec
;
;    Description:
;
;        Reads the sector of FAT in the buffer with offset
;
;    Input:
;
;        eax - offset
;
;    Output:
;
;        bx - offset relatively a sector begin
;
ReadFATSec:

    ; divide offset on bytes count in the sector and get sector number
    ; relatively FAT begin
	movzx	ecx, [bp+B.BytsPerSec]
	div	ecx

	; LBA number of 1st FAT
	add	eax, [bp + B.HiddSec]
	movzx	ecx, [bp + B.RsvdSecCnt]
	add	eax, ecx

	; ensure that 1st FAT is active
	movzx	ebx, [bp + B.ExtFlags]
	and	bx, 0x0f
	jz	.l_01

	; offset on the active FAT
	push	dx
	mov	ecx, eax
	mov	eax, [bp + B.FATSz32]
	mul	ebx
	add	eax, ecx
	pop	dx

	; read sector
	.l_01:
	mov	bx, 0x8000
	mov	cx, 1
	call	ReadSector

	.l_ret:
	mov	bx, dx
	retn

; jump into loaded code
all_ok:
	jmp	0x0060:0x0000

msg_Loader_Not_Found                 : db "Loader not found",0x0
msg_IO_Err                           : db "IO Err",0x0
msg_FS_Err                           : db "FS Err",0x0


times (0x7c00 + 512 - $ - 13) db 0

ProgramName	db "SYSLDR     "

dw	0AA55h
