#!/bin/sh
#
# /etc/rc.d/rc.S:  System initialization script.
#
. /etc/rc.d/rc.conf

PATH=/bin:/usr/bin:/sbin:/usr/sbin

echo -e '\033[1;31;40mSYSTEM INITIALIZATION:\033[0m'

# Start devfsd if necessary
if [ -r /dev/.devfsd ]; then
    if [ -x /sbin/devfsd ]; then
	/sbin/devfsd /dev
    fi
fi

# mounting root device read-only
mount -n -o remount,ro /

# enable swapping
/sbin/swapon -a

# See if a forced filesystem check was requested at shutdown:
if [ -r /etc/forcefsck ]; then
  FORCEFSCK="-f"
fi

# Check the root filesystem:
if [ ! -r /etc/fastboot ]; then
    echo "Checking root filesystem."
    /sbin/fsck $FORCEFSCK -C -a /
    RETVAL=$?
fi

  # An error code of 2 or higher will require a reboot.
  if [ $RETVAL -ge 2 ]; then
    # An error code equal to or greater than 4 means that some errors
    # could not be corrected.  This requires manual attention, so we
    # offer a chance to try to fix the problem in single-user mode:
    if [ $RETVAL -ge 4 ]; then
      echo
      echo
      echo "***********************************************************"
      echo "*** An error occurred during the root filesystem check. ***"
      echo "*** You will now be given a chance to log into the      ***"
      echo "*** system in single-user mode to fix the problem.      ***"
      echo "***                                                     ***"
      echo "*** If you are using the ext2 filesystem, running       ***"
      echo "*** 'e2fsck -v -y <partition>' might help.              ***"
      echo "***********************************************************"
      echo
      echo "Once you exit the single-user shell, the system will reboot."
      echo
      PS1="(Repair filesystem) \#"; export PS1
      sulogin
    else # With an error code of 2 or 3, reboot the machine automatically:
      echo
      echo
      echo "***********************************"
      echo "*** The filesystem was changed. ***"
      echo "*** The system will now reboot. ***"
      echo "***********************************"
      echo
    fi
    echo "Unmounting file systems:"
    umount -a -r
    mount -n -o remount,ro /
    echo "Rebooting system."
    sleep 2
    reboot -f
  fi
  # Remount the root filesystem in read-write mode
  echo "Remounting root device with read-write enabled:"
  /bin/mount -w -v -n -o remount /
if [ $? -gt 0 ] ; then
    echo
    echo "Attempt to remount root device as read-write failed!"
    echo "Please press ENTER to continue, then reboot."
    echo -n "get into your machine and start looking for the problem. " 
    read junk;
fi

# Any /etc/mtab that exists here is old, so we delete it to start over:
/bin/rm -f /etc/mtab*
# Remounting the / partition will initialize the new /etc/mtab:
/bin/mount -w -o remount /

# Initialize the Logical Volume Manager.
# This won't start unless /etc/lvmtab is found, which is created by /sbin/vgscan.
# Therefore, to use LVM you must run /sbin/vgscan yourself the first time.
if [ -r /etc/lvmtab -a "${lvm}" = "YES" ]; then
  echo "Mount /proc early (it's needed for vgscan):"
  /bin/mount -a -t proc
  echo "Scan for new volume groups:"
  /sbin/vgscan
  if [ $? = 0 ]; then
    echo "Make volume groups available to the kernel:"
    /sbin/vgchange -ay
  fi
fi

# Check all the non-root filesystems:
if [ ! -r /etc/fastboot ]; then
  echo "Checking non-root filesystems:"
  /sbin/fsck $FORCEFSCK -C -R -A -a
fi

# mount non-root file systems in fstab (but not NFS or SMB 
# because TCP/IP is not yet configured):
echo "Mounting local filesystems:"
/bin/mount -a -v -t nonfs,nosmbfs

# Clean up some temporary files:
( cd /var/log/setup/tmp && rm -rf * )
/bin/rm -f /var/run/utmp /var/run/*pid /etc/nologin /var/run/lpd* \
  /var/run/ppp* /etc/dhcpc/dhcpcd-eth0.pid /etc/forcefsck /etc/fastboot
/bin/rm -rf /tmp/*

# Create a fresh utmp file:
cat /dev/null > /var/run/utmp

# Set the system time from the hardware clock using hwclock --hctosys.
if [ -x /sbin/hwclock ]; then
  echo "Setting system time."
  /sbin/hwclock --${clock} --hctosys
fi

# This loads any kernel modules that are needed.  These might be required to
# use your ethernet card, sound card, or other optional hardware.
if [ -x /etc/rc.d/rc.modules -a -r /proc/modules ]; then
  . /etc/rc.d/rc.modules
fi

# Run serial port setup script:
# (CAREFUL! This can make some systems hang if the rc.serial script isn't
# set up correctly. If this happens, you may have to edit the file from a
# boot disk)
#
# . /etc/rc.d/rc.serial

# Carry an entropy pool between reboots to improve randomness.
# Load and then save 512 bytes, which is the size of the entropy pool.
if [ -f /etc/random-seed ]; then
  echo "Using /etc/random-seed to initialize /dev/urandom."
  cat /etc/random-seed >/dev/urandom 2>/dev/null
fi
dd if=/dev/urandom of=/etc/random-seed count=1 bs=512 2>/dev/null

